function [FisherMatrix,CRLB] = get_fisher_crlb(PSF,PSFderivatives,parameters)
% This function is for calculating the Fisher matrix and the CRLB given the
% exact vectorial PSF model. The parameters depend on the fitmodel but are
% at most [x,y,z,lambda,N,b] with x,y,z, the coordinates of the emitter,
% lambda the wavelength, N the signal photon count and b the number of
% background photons per pixel. The script can work for a single ROI model:
% size(mu)=[Mx,My] or for multiple ROI model: size(mu)=[Mx,My,Mz].
%
% Sjoerd Stallinga, TU Delft
%

% (C) Copyright 2018
% All rights reserved
% Department of Imaging Physics
% Faculty of Applied Sciences
% Delft University of Technology
% Delft, The Netherlands   

numparams = parameters.numparams;
numders = numparams-2;
Nphindex = numparams-1;
bgindex = numparams;

keps = 1e3*eps;
Nph = parameters.signalphotoncount;
bg = parameters.backgroundphotoncount;
rnvar = (parameters.readnoisestd)^2;
[Mx,My,Mz] = size(PSF);

% calculation of Poisson rates and derivatives
mu = Nph*PSF+bg+rnvar;
mupos = double(mu>0).*mu + double(mu<0)*keps;
weight = 1./mupos;

dmudtheta = zeros(Mx,My,Mz,numparams);
dmudtheta(:,:,:,1:numders) = Nph*PSFderivatives(:,:,:,1:numders);
dmudtheta(:,:,:,Nphindex) = PSF;
dmudtheta(:,:,:,bgindex) = ones(size(PSF));
  
% calculation Fisher matrix
FisherMatrix = zeros(numparams,numparams);
for ii = 1:numparams
  for jj = ii:numparams
    FisherMatrix(ii,jj) = sum(sum(sum(weight.*dmudtheta(:,:,:,ii).*dmudtheta(:,:,:,jj))));
    FisherMatrix(jj,ii) = FisherMatrix(ii,jj);
  end
end
 
% regularization Fisher-matrix in order to circumvent possibility for
% inverting ill-conditioned matrix
if (rcond(FisherMatrix)>keps)
%   CRLB = sqrt(diag(inv(FisherMatrix+keps*eye(size(FisherMatrix)))));
  CRLB = sqrt(diag(inv(FisherMatrix)));
else
  CRLB = zeros(numparams,1);
end
  
end

